using System;
using System.Text;
using System.Collections.Generic;
using Vintagestory.API.Common;
using Vintagestory.API.Common.Entities;
using Vintagestory.API.Config;
using Vintagestory.API.Datastructures;
using Vintagestory.API.MathTools;
using Vintagestory.GameContent;
using Vintagestory.API.Client;
using Vintagestory.API.Server;
using Vintagestory.API.Util;
using System.Runtime;

namespace CraftableCompanion
{

/*	public class EntityBehaviorCraftableCompanionCommandable : EntityBehavior
	{
		public bool Sit {
			get {
				return entity.WatchedAttributes.GetBool ("commandSit", false);
			}
			set {
				entity.WatchedAttributes.SetBool ("commandSit", value);
			}
		}

		public EntityBehaviorCraftableCompanionCommandable (Entity entity)
			: base (entity)
		{
		}

		public override void Initialize (EntityProperties properties, JsonObject attributes)
		{
			base.Initialize (properties, attributes);
		}

		public override void OnInteract (EntityAgent byEntity, ItemSlot itemslot, Vec3d hitPosition, EnumInteractMode mode, ref EnumHandling handled)
		{
			if (byEntity.Controls.Sneak && byEntity.Controls.RightMouseDown && !byEntity.Controls.Sprint){
				Sit = !Sit;
			}
		}

		public override void GetInfoText (StringBuilder infotext)
		{
			base.GetInfoText (infotext);
			if (Sit) {
				infotext.AppendLine (Lang.Get ("Waits", Array.Empty<object> ()));
			} else {
				infotext.AppendLine (Lang.Get ("Follows", Array.Empty<object> ()));
			}
			ITreeAttribute treeAttribute = entity.WatchedAttributes.GetTreeAttribute ("health");
			if (treeAttribute != null) {
				infotext.AppendLine (Lang.Get ("Health: {0:0.##}/{1:0.##}", treeAttribute.GetFloat ("currenthealth", 0f), treeAttribute.GetFloat ("maxhealth", 0f)));
			}
		}

		public override string PropertyName ()
		{
			return "craftablecompanioncommandable";
		}
	}*/

public class EntityBehaviorCraftableCompanionCommandable : EntityBehaviorCommandable
{
    public EntityBehaviorCraftableCompanionCommandable(Entity entity) : base(entity) { }

    /// Custom owner UID, separate from vanilla GuardedName
    public string GuardOwnerUid
    {
        get => entity.WatchedAttributes.GetString("guardOwnerUid", null);
        set => entity.WatchedAttributes.SetString("guardOwnerUid", value);
    }

    /// Like vanilla GetGuardedEntity, but prefers guardOwnerUid if present
    public new Entity GetGuardedEntity()
    {
        if (!string.IsNullOrEmpty(GuardOwnerUid))
        {
            IPlayer player = entity.World.PlayerByUid(GuardOwnerUid);
            return player?.Entity;
        }

        // Fallback to vanilla logic
        return base.GetGuardedEntity();
    }

    public override void OnInteract(EntityAgent byEntity, ItemSlot itemslot, Vec3d hitPosition, EnumInteractMode mode, ref EnumHandling handled)
    {
        // ONLY sneak + RMB toggles sit
        if (byEntity.Controls.Sneak && byEntity.Controls.RightMouseDown && !byEntity.Controls.Sprint)
        {
            Sit = !Sit;
            handled = EnumHandling.PreventDefault;
            return;
        }
    }

    public override string PropertyName() => "craftablecompanioncommandable";
}

public class EntityBehaviorCraftableOpenableContainer : EntityBehavior
{
    protected InventoryGeneric inv;
    protected GuiDialogCreatureContents dlg;
    private WorldInteraction[] interactions;

    public EntityBehaviorCraftableOpenableContainer(Entity entity) : base(entity) { }

    /// Whether only the owner can open this container
    public bool IsLockedToOwner
    {
        get => entity.WatchedAttributes.GetBool("isLockedToOwner", false);
        set => entity.WatchedAttributes.SetBool("isLockedToOwner", value);
    }

	private float interactCooldown = 0f;
	private const float interactCooldownDuration = 1.5f; // seconds

    private void Inv_SlotModified(int slotid)
    {
        TreeAttribute treeAttribute = new TreeAttribute();
        inv.ToTreeAttributes(treeAttribute);
        entity.WatchedAttributes["harvestableInv"] = treeAttribute;
        entity.WatchedAttributes.MarkPathDirty("harvestableInv");
    }

    public override void Initialize(EntityProperties properties, JsonObject typeAttributes)
    {
        inv = new InventoryGeneric(
            typeAttributes["quantitySlots"].AsInt(12),
            "contents-" + entity.EntityId,
            entity.Api,
            null
        );

        TreeAttribute treeAttribute = entity.WatchedAttributes["harvestableInv"] as TreeAttribute;
        if (treeAttribute != null)
        {
            inv.FromTreeAttributes(treeAttribute);
        }

        inv.PutLocked = false;

        if (entity.World.Side == EnumAppSide.Server)
        {
            inv.SlotModified += Inv_SlotModified;
        }

        base.Initialize(properties, typeAttributes);
    }

	public override void OnInteract(EntityAgent byEntity, ItemSlot itemslot, Vec3d hitPosition, EnumInteractMode mode, ref EnumHandling handled)
	{
	    EntityPlayer entityPlayer = byEntity as EntityPlayer;
	    if (entityPlayer == null) return;

	    string playerUid = entityPlayer.PlayerUID;
	    var guardBehavior = entity.GetBehavior<EntityBehaviorCraftableCompanionCommandable>();
	    string ownerUid = guardBehavior?.GuardOwnerUid;
	    //string ownerUid = "00000000-0000-0000-0000-000000000000";

	    //entity.Api.Logger.Debug($"[CompanionDebug] OnInteract called by {playerUid}, item: {itemslot?.Itemstack?.Collectible?.Code?.Path}");

	    // --- Padlock Toggle Logic ---
	    if (!itemslot.Empty && itemslot.Itemstack != null && itemslot.Itemstack.Collectible.Code.Path.Contains("padlock-"))
	    {
			if (this.interactCooldown > 0f)
            {
                return; // to stop multiple sounds.
            }
	        //entity.Api.Logger.Debug($"[CompanionDebug] Padlock interaction detected by {playerUid}");

	        if (string.IsNullOrEmpty(ownerUid))//this should be set by now but better safe than sorry.
	        {
	            //entity.Api.Logger.Debug($"[CompanionDebug] No owner set yet, assigning {playerUid} as owner");
	            guardBehavior.GuardOwnerUid = playerUid;
	            IsLockedToOwner = true;
	        }
	        else if (ownerUid == playerUid)
	        {
	            IsLockedToOwner = !IsLockedToOwner;

	            entity.World.PlaySoundAt(new AssetLocation("craftablecompanion:sounds/locktoggle"), entity);
	            interactCooldown = interactCooldownDuration;//to stop multiple sounds.

	            //entity.Api.Logger.Notification( $"[CompanionDebug] Lock toggled by {playerUid}. New state: {IsLockedToOwner}");
	        }
	        else
	        //{
	        //    entity.Api.Logger.Notification(
	        //        $"[CompanionDebug] Unauthorized lock attempt by {playerUid}. Current owner is {ownerUid}");
	        //}

	        handled = EnumHandling.PreventDefault;
	        return;
	    }

	    // --- Storage Access Logic ---
	    bool inRangeClient = byEntity.World.Side == EnumAppSide.Client && byEntity.Pos.SquareDistanceTo(entity.Pos) <= 5f;
	    bool inRangeServer = byEntity.World.Side == EnumAppSide.Server && byEntity.Pos.SquareDistanceTo(entity.Pos) <= 14f;

	    if ((inRangeClient || inRangeServer) && byEntity.Controls.RightMouseDown && !byEntity.Controls.Sneak)
	    {
	        //entity.Api.Logger.Debug($"[CompanionDebug] {playerUid} attempting to open storage. Locked: {IsLockedToOwner}, Owner: {ownerUid}");

	        if (IsLockedToOwner)
	        {
	            if (!string.IsNullOrEmpty(ownerUid) && ownerUid != playerUid)
	            {
	                //entity.Api.Logger.Notification( $"[CompanionDebug] Access denied for {playerUid}. Storage is locked to owner {ownerUid}");

	                (byEntity.World.Api as ICoreClientAPI)?.TriggerIngameError(this, "notowner", Lang.Get("This companion's storage is locked to its owner."));
	                handled = EnumHandling.PreventDefault;
	                return;
	            }

	            //entity.Api.Logger.Debug($"[CompanionDebug] {playerUid} is owner, access granted.");
	        }
	        else
	        {
	            //entity.Api.Logger.Debug($"[CompanionDebug] Storage unlocked, {playerUid} access granted.");
	        }

	        IPlayer player = entity.World.PlayerByUid(playerUid);
	        player.InventoryManager.OpenInventory(inv);

			// Throttle only the open sound spam
			if (entity.World.Side == EnumAppSide.Server && interactCooldown <= 0f)
			{
			    entity.World.PlaySoundAt(new AssetLocation("craftablecompanion:sounds/openstorage"), entity);
			    interactCooldown = interactCooldownDuration;
			}

           

	        if (entity.World.Side == EnumAppSide.Client && dlg == null)
	        {
	            dlg = new GuiDialogCreatureContents(inv, entity as EntityAgent, entity.Api as ICoreClientAPI, "Companion Storage");
	            if (dlg.TryOpen())
	            {
	                (entity.World.Api as ICoreClientAPI).Network.SendPacketClient(inv.Open(player));
	            }

	            dlg.OnClosed += delegate
	            {
	                dlg.Dispose();
	                dlg = null;
	            };
	        }
	    }
	}


    public override void OnReceivedClientPacket(IServerPlayer player, int packetid, byte[] data, ref EnumHandling handled)
    {
        if (packetid < 1000)
        {
            inv.InvNetworkUtil.HandleClientPacket(player, packetid, data);
            handled = EnumHandling.PreventSubsequent;
        }
        else if (packetid == 1012)
        {
            player.InventoryManager.OpenInventory(inv);
        }
    }

    public override WorldInteraction[] GetInteractionHelp(IClientWorldAccessor world, EntitySelection es, IClientPlayer player, ref EnumHandling handled)
    {
        interactions = ObjectCacheUtil.GetOrCreate(world.Api, "entityContainerInteractions", () => new WorldInteraction[1] {
            new WorldInteraction {
                ActionLangCode = "Access Storage",
                MouseButton = EnumMouseButton.Right,
            }
        });
        return interactions;
    }

    public override void GetInfoText(StringBuilder infotext)
    {
        base.GetInfoText(infotext);

        if (IsLockedToOwner)
        {
            infotext.AppendLine(Lang.Get("Internal Storage: Locked to owner"));
        }
        else
        {
            infotext.AppendLine(Lang.Get("Internal Storage: Unlocked"));
        }
    }

	public override void OnGameTick(float deltaTime)
	{
	    base.OnGameTick(deltaTime);

	    // You can always grab the API from the entity’s world
	    var api = entity.World.Api;

	    if (api.Side == EnumAppSide.Server && interactCooldown > 0f)
	    {
	        interactCooldown -= deltaTime;
	        if (interactCooldown < 0f) interactCooldown = 0f;
	    }

	    // Auto-unlock storage if entity is dead
	    if (!entity.Alive && IsLockedToOwner)
	    {
	        IsLockedToOwner = false;
	    }
	}



    public override string PropertyName() => "craftablecompanionopenablecontainer";
}


}